/**
 * \file: errmem_lib.c
 *
 * Implementation of the main function of the error memory daemon.
 *
 * This file implements the main function of the error memory daemon.
 * This includes the parsing of the command line options, the handling
 * of the backends and the implementation of the server interface.
 *
 * \component: errmemd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *          Markus Kretschmann (mkretschmann@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 * <history item>
 */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/epoll.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/un.h>
#include <semaphore.h>
#include <errno.h>
#include <syslog.h>
#include <linux/errmem.h>
#include "errmem_lib.h"
#include "errmem_lib_init.h"
#include "errmem_socket_interface.h"

#define EXPECTED      10
#define SOCKET_NAME    "/org/adit/errmemd"
#define SOCKET_PATH    ("\0" SOCKET_NAME)
#define MAX_SEM_RETRY 1000

/**
 * struct ClientConnect - Connection information
 * @sid:    Socket ID
 * @e:      epoll dedicated file descriptor
 * @ext_hdl:Connection ID
 * @next:   Next connection (linked list)
 *
 * Structure used to maintain a list of client connection to the errmemd.
 */
struct ClientConnect {
    int sid;
    int e;
    /* private: internal use only */
    int ext_hdl;
    struct ClientConnect *next;
};

/**
 * struct Client - Accessor to client list
 * @ct:     Client count
 * @con:    Head of the client list
 */
static struct Client {
    int ct;
    struct ClientConnect *con;
} cl;

static sem_t sem;

int init(void)
{
    int rc = 0;
    openlog(NULL, LOG_PERROR, 0);

    rc = sem_init(&sem, 0, 1);

    if (-1 == rc)
    {
        syslog(LOG_CRIT, "%s  %m", "semaphore not created:");
    }

    return rc;
}

static void *get_mem(uint32_t size)
{
    void *p = calloc(1, size);

    if (!p)
    {
        syslog(LOG_CRIT, "%s %d", "cannot allocate memory ", size);
    }

    return p;
}

static int32_t get_sem(void)
{
    int32_t rc = 0;
    int32_t again = 1;

    while (again)
    {
        rc = sem_trywait(&sem);

        if (!rc)
        {
            again = 0;
        }
        else if (EAGAIN == errno)
        {
            if (++again > MAX_SEM_RETRY)
            {
                again = 0;
                rc = -EBUSY;
            }
        }
        else
        {
            again = 0;
            rc = -errno;
        }
    }

    if (rc < 0)
    {
        syslog(LOG_CRIT,
               "%s %s", "get_sem: cannot retrieve semaphore - errno = ",
               strerror(-rc));
    }

    return rc;
}

/**
 * create_client_socket() - Create a new connection to errmemd
 * @cc: Internal structure containing connection information
 *
 * Return:  0 on success -errno otherwise
 */
/* PRQA: Lint Message 26: False positive, lint do not recognize the builtin function */
/*lint -save -e26 */
static int32_t create_client_socket(struct ClientConnect *cc)
{
    int32_t rc = socket(PF_UNIX, SOCK_STREAM, 0);
    socklen_t size =
        sizeof(SOCKET_NAME) + __builtin_offsetof(struct sockaddr_un, sun_path);
    struct sockaddr_un addr = { .sun_family = AF_UNIX };

    if (rc < 0)
    {
        return -errno;
    }

    memcpy(addr.sun_path, SOCKET_PATH, sizeof(SOCKET_PATH));
    cc->sid = rc;

    rc = 0;

    if (connect(cc->sid, (struct sockaddr *)&addr, size) < 0)
    {
        close(cc->sid);
        cc->sid = ~0;
        rc = -errno;
    }

    return rc;
}
/*lint -restore */

/**
 * add_client_connect - Add client connection to the list
 * @cc: Client connection to be added
 *
 * The client connection is added at the end of the list.
 */
static void add_client_connect(struct ClientConnect *cc)
{
    struct ClientConnect **cur = &cl.con;

    get_sem();

    cc->ext_hdl = ++cl.ct;

    while (cc && *cur)
    {
        cur = &(*cur)->next;
    }

    if (cc)
    {
        *cur = cc;
    }

    sem_post(&sem);
}

/**
 * remove_client_connect - Remove a client connection from the list
 * @cc: Client connection to be removed
 *
 * Remove a client connection from the client list if it finds it.
 */
static void remove_client_connect(struct ClientConnect *cc)
{
    struct ClientConnect **c = &cl.con;

    while (cc && *c)
    {
        if (*c == cc)
        {
            *c = (*c)->next;
            break;
        }
        else
        {
            c = &(*c)->next;
        }
    }
}

/**
 * get_client_connect_locked - Find a client connection in the list
 * @sh: Connection ID to find
 *
 * This function looks in the list for the connection that has the provided ID.
 * The client list lock is expected to be taken while entering in this function.
 */
static struct ClientConnect *get_client_connect_locked(int sh)
{
    struct ClientConnect *cc = NULL;

    if (sh > 0)
    {
        cc = cl.con;

        while (cc)
        {
            if (cc->ext_hdl == sh)
            {
                break;
            }
            else
            {
                cc = cc->next;
            }
        }
    }

    return cc;
}

/**
 * get_client_connect - Find a client connection in the list
 * @sh: Connection ID to find
 *
 * This function looks in the list for the connection that has the provided ID
 */
static struct ClientConnect *get_client_connect(int sh)
{
    struct ClientConnect *cc = NULL;

    if (get_sem() != 0)
    {
        return cc;
    }

    cc = get_client_connect_locked(sh);

    sem_post(&sem);

    return cc;
}

/**
 * destroy_client_socket - Destroy a client connection
 * @cc: The connection to be destroyed
 *
 * This function will stop polling and destroy resources used by
 * the targeted connection.
 * The structure is not freed there as the create_client_socket is
 * not allocating it.
 */
static void destroy_client_socket(struct ClientConnect *cc)
{
    if (!cc)
    {
        return;
    }

    if (~cc->e)
    {
        if (epoll_ctl(cc->e, EPOLL_CTL_DEL, cc->sid, NULL) < 0)
        {
            syslog(LOG_CRIT,
                   "[%d] EPOLL_CTL_DEL failed (%s)",
                   cc->sid,
                   strerror(errno));
        }

        close(cc->e);
    }

    if (~cc->sid)
    {
        close(cc->sid);
    }
}

#define ERRMEM_LIB_SEND     0
#define ERRMEM_LIB_RECEIVE  1

/**
 * transfer_socket_data - Transfer data through the client socket
 * @p:          Pointer to the data to be sent
 * @len:        Length of the data to be sent
 * @cc:         Client connection to retrieve the socket from
 * @direction:  Either the data shall be sent or received
 *
 * This function tries to receive/send len byte of data through the socket
 * retrieved from the cc client connection.
 *
 * Return: ERR_OK in case of success, error code otherwise
 */
static int32_t transfer_socket_data(void *p,
                                    int32_t len,
                                    struct ClientConnect *cc,
                                    int direction)
{
    char *data = (char *)p;
    int32_t count = 0;
    int32_t rc = 0;

    /* check for data on the socket */
    if (!p || !cc || !(len > 0))
    {
        syslog(LOG_CRIT, "%s: invalid parameter", __func__);
        return -EINVAL;
    }

    do
    {
        if (direction == ERRMEM_LIB_SEND)
        {
            rc = send(cc->sid, data + count, len, 0);
        }
        else
        {
            rc = recv(cc->sid, data + count, len, 0);

            if (rc == 0)
            {
                /* Specific error case for recv */
                rc = -EPIPE;
                break;
            }
        }

        if (rc < 0)
        {
            if ((errno != EAGAIN) && (errno != EWOULDBLOCK))
            {
                rc = -errno;
                break;
            }
            else
            {
                /* Same player play again */
                continue;
            }
        }

        count += rc;
        len -= count;

        if (len < 0)
        {
            rc = -EMSGSIZE;
            break;
        }
        else
        {
            rc = ERR_OK;
        }
    }
    while (len > 0);

    if (rc)
    {
        syslog(LOG_CRIT,
               "[%d] %s - transferred %d - out of %d - err %d",
               cc->sid,
               __func__,
               count,
               len,
               rc);
    }

    return rc;
}

/**
 * send_socket_data - Send data through the client socket
 * @p:      Pointer to the data to be sent
 * @len:    Length of the data to be sent
 * @cc:     Client connection to retrieve the socket from
 *
 * This function tries to send len byte of data through the socket retrieved
 * from the cc client connection.
 *
 * Return: ERR_OK in case of success, error code otherwise
 */
static int32_t send_socket_data(void *p, int32_t len, struct ClientConnect *cc)
{
    return transfer_socket_data(p, len, cc, ERRMEM_LIB_SEND);
}

/**
 * recv_socket_data - Receive data from the client socket
 * @p:      Pointer to the data to be sent
 * @len:    Length of the data to be sent
 * @cc:     Client connection to retrieve the socket from
 *
 * This function tries to receive len byte of data through the socket retrieved
 * from the cc client connection.
 *
 * Return: ERR_OK in case of success, error code otherwise
 */
static int32_t recv_socket_data(void *p, int32_t len, struct ClientConnect *cc)
{
    return transfer_socket_data(p, len, cc, ERRMEM_LIB_RECEIVE);
}

/**
 * check_answer_and_proceed() - Check and process answer for a request
 * @cc:     Client connection to retrieve the socket from
 * @buf:    Buffer to put data in if any
 * @req:    The kind of request that was sent
 *
 * Depending on the request, this function will check if the answer is correct
 * and potentially try to receive the second part of the answer.
 *
 * Return: 0 on success, error code otherwise
 */
static int32_t check_answer_and_proceed(struct ClientConnect *cc,
                                        struct errmem_message *buf,
                                        enum msg_type req)
{
    int rc = 0;
    struct epoll_event event = { 0 };
    struct msg_header pac = {
        .type = req,
        .status = 0,
        .len = 0
    };

    /* The server shall have sent a message already */
    /* FIXME: Infinite wait ????? */
    if (epoll_wait(cc->e, &event, 1, -1) < 0)
    {
        rc = -errno;
        goto out;
    }

    rc = recv_socket_data(&pac, sizeof(pac), cc);

    if ((pac.type == req) && (pac.status >= ERR_OK))
    {
        rc = 0;

        if (req == MSG_REQUEST_READ)
        {
            rc = pac.status;

            if (pac.len > sizeof(pac))
            {
                rc = recv_socket_data(buf, (pac.len - sizeof(pac)), cc);

                if (rc == 0)
                {
                    rc = pac.status;
                }
            }

        }
    }
    else
    {
        rc = pac.status;
    }

out:
    return rc;
}

/**
 * errmem_lib_establish_connection - Establish the connection with errmemd
 * @cc: Client connection to handle
 *
 * This function will check that the connection with errmemd
 * is correctly established.
 *
 * Return: 0 on success, error code otherwise.
 */
static int32_t errmem_lib_establish_connection(struct ClientConnect *cc)
{
    int rc = 0;

    rc = check_answer_and_proceed(cc, NULL, MSG_REQUEST_ESTABLISH);

    if (rc < 0)
    {
        goto out;
    }

    add_client_connect(cc);

out:
    return rc;
}

/**
 * errmem_lib_prepare_polling - Prepare the connection for polling
 * @cc: Client connection
 *
 * This function creates the socket and prepare the epoll structure.
 *
 * Return: 0 on success, error code otherwise.
 */
static int32_t errmem_lib_prepare_polling(struct ClientConnect *cc)
{
    int rc = 0;
    struct epoll_event ev = { .events = EPOLLIN };

    rc = create_client_socket(cc);

    if (rc)
    {
        goto out;
    }

    cc->e = epoll_create(EXPECTED);

    if (cc->e < 0)
    {
        rc = -errno;
        goto out;
    }

    if (epoll_ctl(cc->e, EPOLL_CTL_ADD, cc->sid, &ev) < 0)
    {
        rc = -errno;
        goto out;
    }

    /* As far as the socket is open, the server shall send a message
     * confirming the operation
     */
    rc = errmem_lib_establish_connection(cc);

    if (rc >= 0)
    {
        rc = cc->ext_hdl;
    }

out:
    if (rc < 0)
    {
        destroy_client_socket(cc);
        syslog(LOG_CRIT, "%s - errno: %s", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_open_session - Open a session with the error memory backend
 *
 * The default persistent storage will be assigned
 * automatically to the session.
 *
 * Return: 0 valid session handle to use, error code otherwise
 */
int errmem_backend_open_session(void)
{
    int rc = -ENOLCK;

    struct ClientConnect *cc = (struct ClientConnect *)
        get_mem(sizeof(struct ClientConnect));

    if (!cc)
    {
        rc = -ENOMEM;
        goto out;
    }

    cc->e = ~0;
    cc->sid = ~0;

    if (g_init)
    {
        goto out;
    }

    /* Let's open the socket an prepare the polling */
    rc = errmem_lib_prepare_polling(cc);

out:

    if (rc < 0)
    {
        free(cc);
        syslog(LOG_CRIT, "%s - errno: %s", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_close_session - Close a session with errmemd
 * @sh: session ID received from open call
 *
 * This function will try to find the target client connection in the client
 * list, remove it and destroy the structure.
 *
 * Return: 0 on success, error code otherwise
 */
int errmem_backend_close_session(int sh)
{
    int32_t rc = 0;
    struct ClientConnect *cc = NULL;

    rc = get_sem();

    if (!rc)
    {
        cc = get_client_connect_locked(sh);

        if (cc)
        {
            remove_client_connect(cc);

            if (sem_post(&sem) < 0)
            {
                rc = -errno;
            }

            destroy_client_socket(cc);
            free(cc);
        }
        else
        {
            if (sem_post(&sem) < 0)
            {
                rc = -errno;
            }

            rc = -EINVAL;
        }
    }

    if (rc < 0)
    {
        syslog(LOG_CRIT, "%s failed (%s)", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_send_request() - Send a request to errmemd
 * @sh:     The connection ID
 * @status: The status value to put in the message header
 * @buf:    Buffer where to put the answer if relevant
 * @req:    The request to send.
 *
 * This function is commonly in order to communicate with errmemd.
 *
 * Return: 0 on success, error code otherwise
 */
static int errmem_backend_send_request(int sh,
                                       int status,
                                       struct errmem_message *buf,
                                       enum msg_type req)
{
    int32_t rc = 0;
    struct ClientConnect *cc = NULL;
    struct msg_header pac = {
        .type = req,
        .status = status,
        .len = sizeof(pac)
    };

    if ((req == MSG_REQUEST_READ) && (buf == NULL))
    {
        rc = -EINVAL;
        goto out;
    }

    cc = get_client_connect(sh);

    if (!cc)
    {
        rc = -EINVAL;
        goto out;
    }

    rc = send_socket_data(&pac, pac.len, cc);

    if (rc)
    {
        syslog(LOG_CRIT,
               "send_socket_data failed (%s)",
               strerror(-rc));

        goto out;
    }

    rc = check_answer_and_proceed(cc, buf, req);

out:
    if (rc < 0)
    {
        syslog(LOG_CRIT,
               "%s failed (%s)",
               __func__,
               strerror(-rc));

    }

    return rc;

}

/**
 * errmem_backend_read() - Read a message from a storage.
 * @sh : session handle received from open call
 * @buf: points to memory area where to store the message. This memory
 *       has to be provided and freed by the caller.
 *
 * One call Reads exactly one message from the assigned persistent storage,
 * starting with the oldest one up to the latest one.
 *
 * Return: 0 on success, error code otherwise
 */

int errmem_backend_read(int sh, struct errmem_message *buf)
{
    int32_t rc = 0;

    if (!buf)
    {
        rc = -EINVAL;
        goto out;
    }

    rc = errmem_backend_send_request(sh, 0, buf, MSG_REQUEST_READ);

out:

    if (rc < 0)
    {
        syslog(LOG_CRIT, "%s failed (%s)", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_erase() - Erases the content of a persistent storage
 * @sh: session handle received from open call
 * @ps: number of persistent storage (sequence number
 *      in the list of persistent storages from the
 *      command line which was used to start the daemon)
 *      0: content of all configured persistent storages are erased.
 *      > 0: number of persistent storage which content shall be erased
 *
 * Return: 0 on success, error code otherwise
 */
int errmem_backend_erase(int sh, int ps)
{
    int32_t rc = 0;

    if ((sh > 0) && (ps >= 0))
    {
        rc = errmem_backend_send_request(sh, ps, NULL, MSG_REQUEST_ERASE);
    }
    else
    {
        rc = -EINVAL;
        syslog(LOG_CRIT, "%s failed (%s)", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_set_storage() - Set storage to use
 * @sh: session handle received from open call
 * @ps: number of persistent storage (sequence number
 *      in the list of persistent storages from the
 *      command line which was used to start the daemon)
 *      0: content of all configured persistent storages are erased.
 *      > 0: number of persistent storage which content shall be erased
 *
 * Specify a persistent storage to use in this session other than the current
 * one. It deletes the current assignment and assigns the requested one.
 *
 * Return: 0 on success, error code otherwise
 */
int errmem_backend_set_storage(int sh, int ps)
{
    int32_t rc = 0;

    if ((sh > 0) && (ps >= 0))
    {
        rc = errmem_backend_send_request(sh, ps, NULL, MSG_REQUEST_SET_STORAGE);
    }
    else
    {
        rc = -EINVAL;
        syslog(LOG_CRIT, "%s failed (%s)", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_dump_out() - Dump storage to all outputs
 * @sh: session handle received from open call
 * @out: Where to output
 *
 * Instructs the error memory backend (daemon) to dump the current content
 * of the error memory frontend to all persistent output device like e.g.
 * DLT .
 *
 * Return: 0 on success, error code otherwise
 */
int errmem_backend_dump_out(int sh, int out)
{
    int32_t rc = 0;

    if ((sh > 0) && (out & ERRMEM_ALL_OUT))
    {
        rc = errmem_backend_send_request(sh, out, NULL, MSG_REQUEST_OUT_DEV);
    }
    else
    {
        rc = -EINVAL;
        syslog(LOG_CRIT, "%s failed (%s)", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_dump() - Dump storage to all outputs
 * @sh: session handle received from open call
 * @out: Where to output
 *
 * Instructs the error memory backend (daemon) to dump the current content
 * of the error memory frontend to all persistent output device like e.g.
 * DLT .
 *
 * Return: 0 on success, error code otherwise
 */
int errmem_backend_dump(int sh, int out)
{
    int32_t rc = 0;

    if ((sh > 0) && !(out & ~ERRMEM_ALL_OUT))
    {
        rc = errmem_backend_send_request(sh, out, NULL, MSG_REQUEST_DUMP);
    }
    else
    {
        rc = -EINVAL;
        syslog(LOG_CRIT, "%s failed (%s)", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_stop() - Stops the errmemd backend
 * @sh: session handle received from open call
 *
 * Instructs the backend to stop reading data from the frontend as well as
 * to stop writing data to persistent storages. When this function returns
 * it is guaranteed that the backend does not do any further actions.
 *
 * Return: 0 on success, error code otherwise
 */
int errmem_backend_stop(int sh)
{
    int32_t rc = 0;

    if (sh > 0)
    {
        rc = errmem_backend_send_request(sh, 0,  NULL, MSG_REQUEST_STOP);
    }
    else
    {
        rc = -EINVAL;
        syslog(LOG_CRIT, "%s failed (%s)", __func__, strerror(-rc));
    }

    return rc;
}

/**
 * errmem_backend_deinit() - Deinitialize the library
 *
 * Frees common resources and shall be called when the process using this
 * library will stop execution.
 */
void errmem_backend_deinit(void)
{
    sem_destroy(&sem);
    closelog();
}
